function [oprice,Delta, Vega,Gamma, xi,eta] = WB_f(Ft_vec, strike, rf,TTM, k_vec,isgrad,ishess)
%==========================================================================================
%This function computes the black call option price and related derivatives based on a
%Weibull distribution. The forward price FT has the following distribution:
%f(x) =
%exp(-(strike*gamma(1+1/k)/fprice).^k).*gamma(1/k).*(strike*gamma(1+1/k)/fprice).^(k-1)./Ft;
%
%INPUT:
%   Ft_vec: K-by-1 vector of mean/futures values of each Weibull density
%   strike: N-by-1 strike prices of the options
%   rf: risk-free rate
%   TTM: time to maturity of the options (in years)
%   k_vec: K-by-1 vector of k values of each Weibull density
%   isgrad: logical, whether gradients (first derivatives Greeks) 
%           are computed. Default = false
%   ishess: logical, whether hessians (second derivatives Greeks) 
%           are computed. Default = false
%
%OUTPUT:
%   oprice: N-by-K call option prices implied by each Weibull density for
%           each strike
%   Delta,Vega,Gamma,xi,eta: N-by-K matrices of option Greeks
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 


N=length(strike);
K=length(k_vec);
h = 1e-4; %stepsize for central difference numerical derivatives

oprice=zeros(N,K);
if nargin <6
    isgrad = false; ishess=false;
elseif nargin <7 
    ishess=false;
end
    
if isgrad
    Delta=oprice;
    Vega=oprice;
else
    Delta = []; Vega = [];
end
if ishess
   Gamma = oprice; xi = oprice; eta = oprice;
else
   Gamma = []; xi =[]; eta = [];
end


for i=1:K
    Ft=Ft_vec(i);
    k=k_vec(i);
    sx = (strike*gamma(1+k)/Ft).^(1/k);
    Nd1 = gammainc(sx,1+k,'upper');
    Nd2 = exp(-sx);
    opricek = exp(-rf*TTM)*(Ft.*Nd1 - strike.*Nd2);
    oprice(:,i)=opricek;
    if isgrad
        Delta(:,i) = exp(-rf*TTM).*Nd1;
        kp=k+h;
        km=k-h;
        sx_p = (strike*gamma(1+kp)/Ft).^(1/kp);
        sx_m = (strike*gamma(1+km)/Ft).^(1/km);
        Nd1p = gammainc(sx_p,1+kp,'upper');
        Nd2p = exp(-sx_p);
        Nd1m = gammainc(sx_m,1+km,'upper');
        Nd2m = exp(-sx_m);
        opricep = exp(-rf*TTM)*(Ft.*Nd1p - strike.*Nd2p);
        opricem = exp(-rf*TTM)*(Ft.*Nd1m - strike.*Nd2m);
        Vega(:,i) = (opricep-opricem)/2/h;
        if ishess
            Gamma(:,i) = exp(-rf*TTM)*(exp(-sx).*sx.*strike/Ft^2/k);
            xi(:,i) = (opricep-2*opricek+opricem)/h^2;
            eta(:,i) = exp(-rf*TTM).*(Nd1p-Nd1m)/2/h;
        end
    end
end
end

